const mongoose = require("mongoose");
const Deposit  = require("./models/deposit");
const User     = require("./models/users");


exports.updateUserROIs = async () => {
  const session = await mongoose.startSession();
  session.startTransaction();

  try {
    // Fetch all approved deposits with remaining tenure
    const approvedDeposits = await Deposit.find({
      status: "approved",
      investmentTenure: { $gt: 0 }
    }).session(session);

    for (const deposit of approvedDeposits) {
      const { userId, amount, investmentTenure } = deposit;

      if (investmentTenure <= 0) continue;   // safeguard

      // ─── Fixed ROI percentage bands ───────────────────────────────
      let roiPercentage = 0;
      if (amount >= 500 && amount <= 4_999) {
        roiPercentage = 0.035;        // 3.5 %
      } else if (amount >= 5_000 && amount <= 19_999) {
        roiPercentage = 0.05;         // 5 %
      } else if (amount >= 20_000 && amount <= 49_999) {
        roiPercentage = 0.07;         // 7 %
      } else if (amount >= 50_000) {
        roiPercentage = 0.10;         // 10 %
      }

      // Daily ROI based on the fixed percentage
      const dailyROI = amount * roiPercentage;

      // Update user earnings and deposit tenure atomically
      const user = await User.findById(userId).session(session);
      if (!user) {
        console.warn(`User not found for deposit ID ${deposit._id}`);
        continue;
      }
      
      if (user) {
        user.earnings += dailyROI;
        // deposit.investmentTenure -= 1;

        await user.save({ session });
        // await deposit.save({ session });
      }
    }

    await session.commitTransaction();
    console.log("User ROI and investment tenure updated successfully.");
  } catch (error) {
    await session.abortTransaction();
    console.error("Error updating user ROI and investment tenure:", error);
  } finally {
    session.endSession();
  }
};

exports.decrementInvestmentTenureDaily = async () => {
  const session = await mongoose.startSession();
  session.startTransaction();

  try {
    // Get all deposits with tenure > 0 and approved
    const depositsToUpdate = await Deposit.find({
      status: "approved",
      investmentTenure: { $gt: 0 }
    }).session(session);

    for (const deposit of depositsToUpdate) {
      deposit.investmentTenure -= 1;

      await deposit.save({ session });
    }

    await session.commitTransaction();
    console.log("Daily investmentTenure decremented successfully.");
  } catch (error) {
    await session.abortTransaction();
    console.error("Error decrementing investmentTenure:", error);
  } finally {
    session.endSession();
  }
};
