const bcryptjs = require("bcryptjs");
const jwt = require("jsonwebtoken");
const { v4 } = require("uuid");
const { errorHandler } = require("../utils/error.js");
const User = require("../models/users.js");
const userWallet = require("../models/userWallet.js");

const { sendWelcomeEmail, forgotPasswordMail } = require("../utils/notification.js");
const crypto = require('crypto');
const Deposit = require("../models/deposit.js");
const mongoose = require("mongoose");
const getDownlinesByLevel = require("../utils/getDownlines.js");

require('dotenv').config(); 

exports.signup = async (req, res, next) => {
  try {
    const { 
      firstName, 
      lastName, 
      email, 
      phone, 
      country, 
      admin, 
      password, 
      transactionPin, 
      sponsorId 
    } = req.body;

    // 1. Check if user already exists
    const existingUser = await User.findOne({ email: email.toLowerCase() });
    if (existingUser) {
      return res.status(400).json({ message: "Email already exists. Kindly login." });
    }

    // 2. Hash password + generate OTP
    const hashedPassword = bcryptjs.hashSync(password, 10);
    const otp = crypto.randomInt(1000, 9999).toString();

    let sponsor = null;

    // 3. Validate sponsor (if provided)
    if (sponsorId) {
      sponsor = await User.findOne({ referralId: sponsorId });

      if (!sponsor) {
        return res.status(400).json({ message: "Invalid referral ID." });
      }

      // Prevent self-referral (VERY important)
      if (sponsor.email === email.toLowerCase()) {
        return res.status(400).json({ message: "You cannot refer yourself." });
      }
    }

    // 4. Create the new user
    const newUser = await User.create({
      firstName,
      lastName,
      email: email.toLowerCase(),
      phone,
      country,
      otp,
      password: hashedPassword,
      admin,
      transactionPin,
      referredBy: sponsor ? sponsor.referralId : null
    });

    // 5. Auto-create wallet for this user
    await userWallet.create({
      userId: newUser._id
    });

    // 6. Update sponsor's directReferralCount if sponsor exists
    if (sponsor) {
      await User.updateOne(
        { _id: sponsor._id },
        { $inc: { directReferralCount: 1 } }
      );
    }

    // 7. Send welcome email (optional)
    await sendWelcomeEmail(newUser);

    return res.status(201).json({
      message: "Account created successfully",
      user: newUser
    });

  } catch (error) {
    console.error("Error creating user:", error);
    next(error);
  }
};

exports.signin = async (req, res, next) => {
  const { email, password } = req.body;
  try {
    const normalizedEmail = email.toLowerCase();
    const validUser = await User.findOne({ email: normalizedEmail });

    if (!validUser) return next(errorHandler(404, "User not found"));

    const validPassword = bcryptjs.compareSync(password, validUser.password);

    if (!validPassword) return next(errorHandler(401, "Wrong credentials"));

    if (!process.env.JWT_SECRET) {
      return next(errorHandler(500, "JWT secret is not defined"));
    }

    // const token = jwt.sign({ id: validUser._id }, process.env.JWT_SECRET);
       const token = jwt.sign(
      {
        id: validUser._id,
        email: validUser.email,
        referralId: validUser.referralId
      },
      process.env.JWT_SECRET,
      { expiresIn: "1d" }
    );

    const { password: pass, ...userData } = validUser.toObject(); // Use toObject() for Mongoose

       res.status(200).json({
      success: true,
      token,          // ✅ SEND TOKEN
      user: userData, // ✅ SEND USER
    });
  } catch (error) {
    next(error);
  }
};

exports.signOut = async (req, res, next) => {
  try {
    res.clearCookie("access_token");
    res.status(200).json("User has been logged out!");
  } catch (error) {
    next(error);
  }
};

exports.getDownlines = async (req, res) => {
  try {
    // 1. Ensure authenticated user exists
    if (!req.user || !req.user.referralId) {
      return res.status(401).json({ message: "Unauthorized request." });
    }

    const referralId = req.user.referralId;

    // 2. Optional level limit (default 15)
    const levels = parseInt(req.query.levels) || 16;

    // 3. Fetch downline tree
    const tree = await getDownlinesByLevel(referralId, levels);

    // 4. Format summary
    const summary = Object.keys(tree).map(level => ({
      level: Number(level),
      count: tree[level].length
    }));

    return res.status(200).json({
      success: true,
      levels: summary,
      tree
    });

  } catch (err) {
    console.error("❌ Error fetching downlines:", err);
    res.status(500).json({ message: "Error fetching downlines" });
  }
};

exports.updateUser = async (req, res, next) => {
  
  if (req.user.id !== req.params.id) {
    return next(errorHandler(401, "You can only update your own account!"));
  }

  try {
    const { firstName, lastName, country, phone, email, password } = req.body;

    let updates = {
      firstName,
      lastName,
      country,
      phone,
      email,
    };

    if (password) {
      updates.password = bcryptjs.hashSync(password, 10);
    }

    const updatedUser = await User.findByIdAndUpdate(req.params.id, updates, { new: true });

    if (!updatedUser) {
      return next(errorHandler(404, "User not found"));
    }

    res.status(200).json(updatedUser);
  } catch (error) {
    next(error);
  }
};

exports.updateTransactionPin = async (req, res, next) => {
  // Ensure the user is updating their own account
  if (req.user.id !== req.params.id) {
    return next(errorHandler(401, "You can only update your own account!"));
  }

  try {
    const { OldTransactionPin, NewTransactionPin } = req.body;

    // Find the user by ID
    const user = await User.findById(req.params.id);
    if (!user) {
      return next(errorHandler(404, "User not found"));
    }

    // Check if the old transaction pin matches
    if (user.transactionPin !== OldTransactionPin) {
      return next(errorHandler(400, "Incorrect old transaction pin"));
    }

    // Update with the new transaction pin
    user.transactionPin = NewTransactionPin;
    await user.save();

    res.status(200).json({ 
      message: "Transaction pin updated successfully", 
      transactionPin: NewTransactionPin 
    });
  } catch (error) {
    next(error);
  }
};

exports.updatePassword = async (req, res, next) => {
  // Ensure the user is updating their own account
  if (req.user.id !== req.params.id) {
    return next(errorHandler(401, "You can only update your own account!"));
  }

  try {
    const { OldPassword, NewPassword } = req.body;

    // Find the user by ID
    const user = await User.findById(req.params.id);
    if (!user) {
      return next(errorHandler(404, "User not found"));
    }

    // Check if the old transaction pin matches
    const isPasswordMatch = await bcryptjs.compare(OldPassword, user.password);
    if (!isPasswordMatch) {
      return next(errorHandler(400, "Incorrect old password"));
    }

    // Update with the new transaction pin
    user.password = bcryptjs.hashSync(NewPassword, 10);
    await user.save();

    res.status(200).json({ message: "Password updated successfully" });
  } catch (error) {
    next(error);
  }
};

exports.updateUserWallets = async (req, res, next) => {
  if (req.user.id !== req.params.id) {
    return next(errorHandler(401, "You can only update your own account!"));
  }

  try {
    const { btcWallet, usdtWallet, ethWallet, financialPassword } = req.body;

    // Fetch the user to verify the transactionPin
    const user = await User.findById(req.params.id);
    if (!user) {
      return next(errorHandler(404, "User not found"));
    }

    // Check if the provided financialPassword matches the transactionPin
    if (financialPassword !== user.transactionPin) {
      return next(errorHandler(403, "Incorrect transaction pin. Please try again."));
    }

    let updates = {
      btcWallet,
      usdtWallet,
      ethWallet,
    };

    const updatedUser = await User.findByIdAndUpdate(req.params.id, updates, { new: true });

    res.status(200).json(updatedUser);
  } catch (error) {
    next(error);
  }
};

exports.forgotPassword = async (req, res, next) => {
  try {
    const { email } = req.body;
    const normalizedEmail = email.toLowerCase();
    const user = await User.findOne({ email: normalizedEmail });
    
    if (!user) {
      return res.status(404).json({ success: false, message: "User with this email does not exist." });
    }

    await forgotPasswordMail(user);

    return res.status(200).json({ success: true, message: "Password reset link sent" });
  } catch (error) {
    console.error(error); // Log the error for debugging
    return res.status(500).json({
      success: false,
      message: error.message || "Internal Server Error",
    });
  }
};

exports.resetPassword = async (req, res, next) => {
  try {
    const { otp, newPassword } = req.body;

    // Find the user by OTP
    const user = await User.findOne({ otp });

    if (!user) {
      return res.status(400).json({ message: "Invalid or expired OTP" });
    }

    // Generate a new OTP
    const newOTP = crypto.randomInt(1000, 9999).toString();

    // Hash the new password
    const hashedPassword = bcryptjs.hashSync(newPassword, 10);

    // Update the user's password and OTP
    user.password = hashedPassword;
    user.otp = newOTP;
    await user.save();

    return res.status(200).json({ message: "Password reset successfully" });
  } catch (error) {
    next(error);
  }
};

exports.addReferralCommissionToEarnings = async (req, res, next) => {
  try {
    const { userId, referralId } = req.body;

    const user = await User.findById(userId);
    if (!user) return next(errorHandler(404, 'User not found'));

    // Find the referral in the user's referral array
    const referral = user.referrals.find(r => r.referralId === referralId);
    if (!referral) return next(errorHandler(404, 'Referral not found'));

    if (referral.addedToEarnings) {
      return next(errorHandler(400, 'Commission already added to earnings'));
    }

    const commissionAmount = Number(referral.commission);

    // Check if user has enough referral earnings
    if (user.ReferralEarnings < commissionAmount) {
      return next(errorHandler(400, 'Insufficient referral earnings balance'));
    }

    // Update balances
    user.ReferralEarnings -= commissionAmount;
    user.earnings = (user.earnings || 0) + commissionAmount;

    // Mark referral as added
    referral.addedToEarnings = true;

    await user.save();

    res.status(200).json({
      message: 'Commission added to earnings successfully.',
      updatedUser: user,
    });
  } catch (error) {
    next(error);
  }
};

// get user wallet data
exports.userWalletData = async (req, res, next) => {

  try {
    // Ensure authenticated user
    if (!req.user || !req.user.id) {
      return res.status(401).json({ message: "Unauthorized request." });
    }

    const wallet = await userWallet.findOne({
      userId: req.user.id,
    }).select(
      "balance activeBalance referralEarnings earnings totalWithdrawn pendingWithdrawal"
    );

    if (!wallet) {
      return res.status(404).json({
        success: false,
        message: "Wallet not found.",
      });
    }

    return res.status(200).json({
      success: true,
      wallet,
    });
  } catch (err) {
    console.error("❌ Error fetching user wallet:", err);
    next(err);
  }
};


// ADMIN

exports.getAllUsers = async (req, res, next) => {
  try {
    const users = await User.find();

    if (!users || users.length === 0) {
      return res.status(404).json({ message: "No users found" });
    }

    res.status(200).json({ users });
  } catch (error) {
    console.error("Error fetching users:", error);
    res.status(500).json({ message: "Failed to fetch users" });
  }
};

exports.updateUserByAdmin = async (req, res, next) => {
  try {
    const { accountBalance, ROI, pendingWithdrawal, totalWithdrawal, admin } = req.body;

    let updates = {
      accountBalance,
      ROI,
      pendingWithdrawal,
      totalWithdrawal,
      admin,
    };

    console.log('updates', updates)

    const updatedUser = await User.findByIdAndUpdate(req.params.id, updates, { new: true });

    if (!updatedUser) {
      return next(errorHandler(404, "User not found"));
    }

    res.status(200).json(updatedUser);
  } catch (error) {
    next(error);
  }
};

exports.getUserById = async (req, res, next) => {
  const userId = req.params.id;

  try {
    const user = await User.findById(userId);

    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    const { password: pass, ...rest } = user.toObject();

    res.status(200).json(rest);
  } catch (error) {
    console.error("Error fetching user:", error);
    res.status(500).json({ message: "Failed to fetch user" });
  }
};

exports.deleteUser = async (req, res, next) => {
  const userId = req.params.id;

  try {
    // Use findByIdAndDelete to delete the user directly
    const user = await User.findByIdAndDelete(userId);

    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    res.status(200).json({ message: "User deleted successfully" });
  } catch (error) {
    console.error("Error deleting user:", error);
    res.status(500).json({ message: "Failed to delete user" });
  }
};

exports.updateUserROIs = async () => {
  const session = await mongoose.startSession(); // Start session for transaction
  session.startTransaction(); // Start transaction
  try {
    // Fetch all approved deposits
    const approvedDeposits = await Deposit.find({ status: "approved" }).session(session);

    // Process each approved deposit
    for (const deposit of approvedDeposits) {
      const { userId, amount, investmentPlan } = deposit;

      // Determine the ROI percentage based on the investment plan
      let roiPercentage;
      switch (investmentPlan) {
        case "Forex & crypto":
          roiPercentage = 0.006;  // 0.6% daily
          break;
        case "Stock & ETF":
          roiPercentage = 0.003;  // 0.3% daily
          break;
        case "Medical Cannabis":
          roiPercentage = 0.005;  // 0.5% daily
          break;
        case "Real Estate":
          roiPercentage = 0.0008;  // 0.08% daily
          break;
        default:
          roiPercentage = 0; // In case of an unrecognized investment plan
          break;
      }

      // Calculate the daily ROI based on the deposit amount
      const dailyROI = amount * roiPercentage;

      // Find the corresponding user
      const user = await User.findById(userId).session(session); // Query within the transaction

      if (user) {
        // Update the user's ROI field
        user.ROI += dailyROI;
        await user.save({ session }); // Save within the transaction
      }
    }

    // Commit transaction if everything succeeds
    await session.commitTransaction();
    session.endSession(); // End session
    console.log("User ROI updated successfully.");
  } catch (error) {
    // Rollback transaction in case of an error
    await session.abortTransaction();
    session.endSession(); // End session
    console.error("Error updating user ROI:", error);
  }
};


