const RANKS = require("../config/ranks");
const UserWallet = require("../models/userWallet");
const User = require("../models/users");

// 🔹 Rank evaluation (used internally)
async function evaluateRankAndReward(user, session) {
  const stats = user.rankStats;

  if (!Array.isArray(user.rankRewards)) {
    user.rankRewards = [];
  }

  for (const [, rules] of Object.entries(RANKS)) {
    const qualifies =
      stats.personalDeposit >= rules.personalDeposit &&
      stats.directReferralDeposit >= rules.directReferralDeposit &&
      stats.indirectReferralDeposit >= rules.indirectReferralDeposit &&
      stats.groupVolume >= rules.groupVolume;

    if (!qualifies) continue;

    const rankName = rules.name; // 🔑 UI NAME (e.g. "CO-FOUNDER")

    const alreadyRewarded = user.rankRewards.includes(rankName);

    // Always upgrade rank to the highest qualified
    user.rank = rankName;

    if (!alreadyRewarded) {
      const wallet = await UserWallet.findOne(
        { userId: user._id }
      ).session(session);

      if (!wallet) {
        throw new Error("Wallet not found during rank reward");
      }

      wallet.rankRewards += rules.reward;
      await wallet.save({ session });

      user.rankRewards.push(rankName);
    }
  }

  await user.save({ session });
}


// 🔹 API controller
async function getUserRank(req, res, next) {
  try {
    if (!req.user || !req.user.id) {
      return res.status(401).json({ message: "Unauthorized request." });
    }

    const user = await User.findById(req.user.id).select(
      "rank rankStats rankRewards"
    );

    if (!user) {
      return res.status(404).json({ message: "User not found." });
    }

    return res.status(200).json({
      success: true,
      rank: user.rank,
      rankStats: user.rankStats,
      rankRewards: user.rankRewards,
    });
  } catch (err) {
    console.error("❌ Error fetching user rank:", err);
    next(err);
  }
}


// ✅ EXPORT BOTH (THIS IS THE KEY)
module.exports = {
  evaluateRankAndReward,
  getUserRank
};
