const errorHandler = require("../utils/error.js");
const Deposit = require("../models/deposit.js");
const User = require("../models/users.js");
const { sendPaymentInvoice } = require("../utils/notification.js");
const Withdrawal = require("../models/withdrawal.js");
const axios = require("axios");

const API_KEY = process.env.NOWPAYMENTS_API_KEY;

// Fetch available crypto currencies
exports.getCurrencies = async (req, res) => {
  // console.log("getCurrencies")
  try {
    const response = await axios.get('https://api.nowpayments.io/v1/currencies', {
      headers: { 'x-api-key': API_KEY },
    });

    // Filter only enabled crypto currencies
    const cryptoCurrencies = response.data.currencies;
    // console.log('cryptoCurrencies', cryptoCurrencies)

    res.status(200).json({ success: true, data: cryptoCurrencies });
  } catch (err) {
    console.error('❌ Failed to fetch currencies:', err.message);
    res.status(500).json({ success: false, message: 'Failed to fetch currencies' });
  }
};


exports.deposit = async (req, res) => {
  try {
    const { amount, currency } = req.body;
    const userId = req.user.id;

    if (!amount || Number(amount) <= 0) {
      return res.status(400).json({ message: "Invalid deposit amount" });
    }

    if (!currency) {
      return res.status(400).json({ message: "Currency is required" });
    }

    // Unique order reference (important for webhook idempotency)
    const orderId = `DEP_${userId}_${Date.now()}`;

    const payload = {
      price_amount: Number(amount),
      price_currency: "usd",
      pay_currency: currency,
      order_id: orderId,
      order_description: `User deposit`,
      success_url: `${process.env.FRONTEND_URL}/deposit-success`,
      cancel_url: `${process.env.FRONTEND_URL}/deposit-cancel`,
      ipn_callback_url: `${process.env.BACKEND_URL}/api/webhook/nowpayments`,
    };

    const paymentRes = await axios.post(
      "https://api.nowpayments.io/v1/invoice",
      payload,
      {
        headers: {
          "x-api-key": API_KEY,
          "Content-Type": "application/json",
        },
      }
    );

    const invoice = paymentRes.data;
    console.log('invoice', invoice)

    // Save pending deposit
    await Deposit.create({
      userId,
      amount,
      currency: invoice.pay_currency,
      paymentId: invoice.id,
      invoiceUrl: invoice.invoice_url,
      orderId: invoice.order_id,
      // depositAddress: invoice.pay_address,
      status: "pending",
    });

    return res.status(200).json({
      success: true,
      invoiceUrl: invoice.invoice_url,
      orderId,
    });

  } catch (error) {
    console.error("❌ Deposit creation failed:", error.response?.data || error);
    return res.status(500).json({
      success: false,
      message: "Failed to create deposit",
    });
  }
};

// controllers/deposit.controller.js

exports.getDepositStatus = async (req, res) => {
  // console.log('first')
  try {
    const { orderId } = req.params;

    if (!orderId) {
      return res.status(400).json({
        success: false,
        message: "Order ID is required",
      });
    }

    // IMPORTANT: match against paymentId in DB
    const deposit = await Deposit.findOne({ paymentId: orderId }).select(
      "status amount currency createdAt"
    );

    if (!deposit) {
      return res.status(404).json({
        success: false,
        message: "Deposit not found",
      });
    }

    return res.status(200).json({
      success: true,
      status: deposit.status, // pending | completed | failed
      amount: deposit.amount,
      currency: deposit.currency,
      createdAt: deposit.createdAt,
    });
  } catch (err) {
    console.error("Deposit status error:", err);
    return res.status(500).json({
      success: false,
      message: "Unable to fetch deposit status",
    });
  }
};




exports.getUserDeposit = async (req, res, next) => {
  try {
    const userId = req.params.id;

    // Find deposits by userId
    const userDeposits = await Deposit.find({ userId: userId });

    if (!userDeposits || userDeposits.length === 0) {
      return res.status(200).json([]); // Return empty array instead of 404
    }
    

    // Return the found deposits
    res.status(200).json(userDeposits);
  } catch (error) {
    // Handle any errors that occur during the process
    console.error("Error fetching deposit:", error);
    res.status(500).json({ message: "Failed to fetch deposit" });
  }
};


exports.updateTotalBalance = async (req, res, next) => {
  const userId = req.params.id;

  try {
    // Fetch approved deposits for the user
    const approvedDeposits = await Deposit.find({
      userId: userId,
      status: "approved",
    });

    // Fetch approved withdrawals for the user
    const approvedWithdrawals = await Withdrawal.find({
      userId: userId,
      status: "approved",
    });

    // Find the user
    const user = await User.findById(userId);

    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    // Calculate total approved deposit amount
    const totalApprovedDeposits = approvedDeposits.reduce((total, deposit) => {
      return total + deposit.amount;
    }, 0);

    // Calculate total approved withdrawal amount
    const totalApprovedWithdrawals = approvedWithdrawals.reduce((total, withdrawal) => {
      return total + withdrawal.amount;
    }, 0);

    // Update user's account balance and total withdrawal fields
    user.accountBalance = totalApprovedDeposits;
    user.totalWithdrawal = totalApprovedWithdrawals;

    // Save the updated user data
    await user.save();

    return res.status(200).json({
      accountBalance: user.accountBalance,
      totalWithdrawal: user.totalWithdrawal,
    });
  } catch (error) {
    console.error("Error updating total balance and withdrawals:", error);
    return res.status(500).json({ message: "Failed to update total balance and withdrawals" });
  }
};


exports.getAllDeposits = async (req, res, next) => {
  try {
    // Populate the user and select only the 'name' field
    const deposits = await Deposit.find().populate('userId', 'firstName lastName');

    if (!deposits || deposits.length === 0) {
      return res.status(404).json({ message: "No Deposits found" });
    }

    res.status(200).json({ deposits });
  } catch (error) {
    console.error("Error fetching deposits:", error);
    res.status(500).json({ message: "Failed to fetch deposits" });
  }
};


// Get single deposit
exports.getSingleDeposit = async (req, res, next) => {
  const depositId = req.params.id;

  try {
    const deposit = await Deposit.findById(depositId);

    if (!deposit) {
      return res.status(404).json({ message: "Deposit not found" });
    }

    res.status(200).json({ deposit });
  } catch (error) {
    console.error("Error fetching deposit:", error);
    res.status(500).json({ message: "Failed to fetch deposit" });
  }
};


// exports.updateDeposit = async (req, res, next) => {
//   try {
//     const { investmentPlan, amount, paymentMethod, status } = req.body;

//     const existingDeposit = await Deposit.findById(req.params.id);
//     if (!existingDeposit) {
//       return next(errorHandler(404, "Deposit not found"));
//     }

//     const user = await User.findById(existingDeposit.userId);
//     if (!user) {
//       return next(errorHandler(404, "User not found"));
//     }

//     console.log('existingDeposit.status:', existingDeposit.status);
// console.log('new status:', status);


//     const numericAmount = Number(amount);
//     const previousAmount = Number(existingDeposit.amount);
//     user.accountBalance = Number(user.accountBalance) || 0;

//     // ✅ Handle transitions or updates involving 'approved' status
//     if (existingDeposit.status === "pending" && status === "approved") {
//       // Going from pending to approved — add full amount
//       user.accountBalance += numericAmount;
//     } else if (existingDeposit.status === "approved" && status === "pending") {
//       // Going back from approved to pending — remove full previous amount
//       user.accountBalance -= previousAmount;
//     } else if (existingDeposit.status === "approved" && status === "approved" && numericAmount !== previousAmount) {
//       // ✅ Approved to approved and amount changed — adjust by the difference
//       const difference = numericAmount - previousAmount;
//       user.accountBalance += difference;
//     }

//     await user.save();

//     const updates = {
//       investmentPlan,
//       amount: numericAmount,
//       paymentMethod,
//       status,
//     };

//     const updatedDeposit = await Deposit.findByIdAndUpdate(
//       req.params.id,
//       updates,
//       { new: true }
//     );

//     res.status(200).json(updatedDeposit);
//   } catch (error) {
//     next(error);
//   }
// };

exports.updateDeposit = async (req, res, next) => {
  try {
    const { investmentPlan, amount, paymentMethod, status } = req.body;

    const existingDeposit = await Deposit.findById(req.params.id);
    if (!existingDeposit) return next(errorHandler(404, "Deposit not found"));

    const user = await User.findById(existingDeposit.userId);
    if (!user) return next(errorHandler(404, "User not found"));

    console.log('existingDeposit.status:', existingDeposit.status);
    console.log('new status:', status);

    const numericAmount = Number(amount);
    const previousAmount = Number(existingDeposit.amount);
    user.accountBalance = Number(user.accountBalance) || 0;

    // Handle transitions or updates involving 'approved' status
    if (existingDeposit.status === "pending" && status === "approved") {
      // Approving a new deposit
      user.accountBalance += numericAmount;

      // ✅ Handle referral bonus only on first approved deposit
      if (!user.firstDeposit) {
        user.firstDeposit = true;

        // Check if user was referred
        if (user.referredBy !== null) {
          const upline = await User.findOne({ referralId: user.referredBy });

          if (upline) {
            const bonus = 0.05 * numericAmount;

            // Add bonus to upline's referral earnings
            upline.ReferralEarnings = (upline.ReferralEarnings || 0) + bonus;

            // Update matching referral entry
            upline.referrals = upline.referrals.map(ref => {
              if (ref.referralId === user.referralId) {
                return {
                  ...ref,
                  firstDepositAmount: numericAmount,
                  commission: bonus
                };
              }
              return ref;
            });

            await upline.save();
          }
        }
      }
    } else if (existingDeposit.status === "approved" && status === "pending") {
      // Reverting approval
      user.accountBalance -= previousAmount;
    } else if (existingDeposit.status === "approved" && status === "approved" && numericAmount !== previousAmount) {
      // Adjusting approved deposit amount
      const difference = numericAmount - previousAmount;
      user.accountBalance += difference;
    }

    await user.save();

    const updates = {
      investmentPlan,
      amount: numericAmount,
      paymentMethod,
      status,
    };

    const updatedDeposit = await Deposit.findByIdAndUpdate(
      req.params.id,
      updates,
      { new: true }
    );

    res.status(200).json(updatedDeposit);
  } catch (error) {
    next(error);
  }
};


exports.deleteDeposit = async (req, res, next) => {
  const depositId = req.params.id; // Get the deposit ID from the request parameters

  try {
    const deposit = await Deposit.findByIdAndDelete(depositId); // Find the deposit by its ID

    if (!deposit) {
      return res.status(404).json({ message: "Deposit not found" });
    }


    res.status(200).json({ message: "Deposit deleted successfully" });
  } catch (error) {
    console.error("Error deleting deposit:", error);
    res.status(500).json({ message: "Failed to delete deposit" });
  }
};
