const BotPlan = require("../models/botPlan");


exports.createBotPlan = async (req, res, next) => {
  try {
    const {
      name,
      slug,
      description,
      minTradesPerDay,
      maxTradesPerDay,
      dailyRoiPercent,
      minAmount,
      maxAmount,
      durationDays,
      activationFee,
      freeDays,
    } = req.body;

    // Basic validation
    if (
      !name ||
      !slug ||
      minTradesPerDay == null ||
      maxTradesPerDay == null ||
      dailyRoiPercent == null ||
      minAmount == null ||
      maxAmount == null
    ) {
      return res.status(400).json({
        message: "Missing required bot plan fields.",
      });
    }

    if (minTradesPerDay > maxTradesPerDay) {
      return res.status(400).json({
        message: "minTradesPerDay cannot exceed maxTradesPerDay.",
      });
    }

    if (minAmount > maxAmount) {
      return res.status(400).json({
        message: "minAmount cannot exceed maxAmount.",
      });
    }

    // Ensure unique slug
    const existing = await BotPlan.findOne({ slug });
    if (existing) {
      return res.status(400).json({
        message: "Bot plan with this slug already exists.",
      });
    }

    const botPlan = await BotPlan.create({
      name,
      slug,
      description,
      minTradesPerDay,
      maxTradesPerDay,
      dailyRoiPercent,
      minAmount,
      maxAmount,
      durationDays: durationDays || null,
      activationFee: activationFee || 0,
      freeDays: freeDays || 0,
    });

    res.status(201).json({
      success: true,
      message: "Bot plan created successfully.",
      botPlan,
    });
  } catch (err) {
    console.error("Create bot plan error:", err);
    next(err);
  }
};

// GET /api/bots/plans
exports.getBotPlans = async (req, res, next) => {
  try {
    const plans = await BotPlan.find({ isActive: true }).sort({ minAmount: 1 });
    res.status(200).json({ success: true, plans });
  } catch (err) {
    console.error("Error fetching bot plans:", err);
    next(err);
  }
};

/*** UPDATE BOT PLAN* PUT /api/admin/bot-plans/:id*/
exports.updateBotPlan = async (req, res, next) => {
  try {
    const { id } = req.params;
    const updates = req.body;

    const plan = await BotPlan.findById(id);
    if (!plan) {
      return res.status(404).json({
        message: "Bot plan not found.",
      });
    }

    // Prevent invalid updates
    if (
      updates.minTradesPerDay != null &&
      updates.maxTradesPerDay != null &&
      updates.minTradesPerDay > updates.maxTradesPerDay
    ) {
      return res.status(400).json({
        message: "minTradesPerDay cannot exceed maxTradesPerDay.",
      });
    }

    if (
      updates.minAmount != null &&
      updates.maxAmount != null &&
      updates.minAmount > updates.maxAmount
    ) {
      return res.status(400).json({
        message: "minAmount cannot exceed maxAmount.",
      });
    }

    // Slug uniqueness check (if updating slug)
    if (updates.slug && updates.slug !== plan.slug) {
      const existing = await BotPlan.findOne({ slug: updates.slug });
      if (existing) {
        return res.status(400).json({
          message: "Another bot plan already uses this slug.",
        });
      }
    }

    // Apply updates safely
    Object.keys(updates).forEach((key) => {
      plan[key] = updates[key];
    });

    await plan.save();

    res.status(200).json({
      success: true,
      message: "Bot plan updated successfully.",
      botPlan: plan,
    });
  } catch (err) {
    console.error("Update bot plan error:", err);
    next(err);
  }
};


